/*
  // March 2014 - TMRh20 - Updated along with High Speed RF24 Library fork
  // Parts derived from examples by J. Coliz <maniacbug@ymail.com>
*/
#include <SPI.h>
#include "nRF24L01.h"
#include "RF24.h"
#include "printf.h"

// Hardware configuration: Set up nRF24L01 radio on SPI bus plus pins 9 & 10
RF24 radio(9, 10);

int rolePin = 2;
int sensorPin = 8;

// Topology
const uint64_t pipes[3] = { 0xABCDABCD71LL, 0x544d52687CLL, 0xC08C08687CLL };              // Radio pipe addresses for the 2 nodes to communicate.

// Role management: Set up role.  This sketch uses the same software for all the nodes
// in this system.  Doing so greatly simplifies testing.

typedef enum { role_start = 1, role_stop } role_e;                 // The various roles supported by this sketch
const char* role_friendly_name[] = { "invalid", "Start", "Stop"};  // The debug-friendly names of those roles
role_e role = role_stop;                                              // The role of the current running sketch

// A single byte to keep track of the data being sent back and forth
byte counter = 1;

unsigned int Stato = 0;
long int t0 = 0;
long int t1 = 0;

void setup() {

  Serial.begin(115200);

  pinMode(rolePin, INPUT);
  pinMode(sensorPin, INPUT);

  if (digitalRead(rolePin) == HIGH)
    role = role_start;
  else
    role = role_stop;

  printf_begin();
  Serial.println("***************************************");
  Serial.println("       CRONOCURLING FOTOCELLULA");
  Serial.println("***************************************");
  Serial.println("");
  Serial.println("");
  Serial.println(role_friendly_name[role]);

  // Setup and configure rf radio

  radio.begin();
  radio.setAutoAck(1);                    // Ensure autoACK is enabled
  radio.setRetries(0, 3);                // Smallest time between retries, max no. of retries
  if (role == role_start) {
    radio.openWritingPipe(pipes[1]);
    radio.openReadingPipe(1, pipes[0]);
  }
  else {
    radio.openWritingPipe(pipes[0]);
    radio.openReadingPipe(1, pipes[1]);
    radio.openReadingPipe(2, pipes[2]);
  }
  radio.startListening();                 // Start listening
  radio.printDetails();                   // Dump the configuration of the rf unit for debugging

  Stato = 1;
}

void loop(void) {

  if (role == role_start) {

    switch (Stato) {

      case 1:                               //Attesa del pronto da parte della fotocellula di stop (il master)
        attesaPronto();
        Stato = 2;                        //spostato nella funzione precedente a seconda del risultato

        break;

      case 2:                               //Attesa del ok per lo start e invio del segnale di start alla fotocellula di stop
        attesaStart();
        Stato = 1;

        break;
    }
  }
  else {

    switch (Stato) {

      case 1:                               //Pronto
        pronto();
        //Stato = 2;                        //spostato nella funzione precedente a seconda del risultato

        break;

      case 2:                               //Start time
        inviaValoreADisplay(0);
        startTime();
        //Stato = 3;

        break;

      case 3:                               //Stop time
        stopTime();
        //Stato = 4;

        break;

      case 4:                               //Display
        displayTime();
        delay(10000);
        Stato = 1;

        break;

      case 5:                               //Errore
      case 0:
        errore();
        delay(6000);
        Stato = 1;
    }
  }
}

void pronto(void) {

  radio.stopListening();
  Serial.println("*** Fotocellula start é disponibile?");

  unsigned int risposta = 0 ;

  if (!radio.write( &Stato, sizeof(unsigned int) )) {
    Serial.println(">>> ERRORE nel invio dello stato pronto");
    Stato = 5;
    radio.startListening();
  }
  else {

    if (!radio.available()) {
      Serial.println(">>> Ricevuto ACK dalla fotocellula start");
    } else {
      Serial.println(">>> ERRORE, nessun ACK ricevuto dalla fotocellula start");
    }

    radio.startListening();

    byte pipeNo;
    bool exitLoop = false;
    int count = 0;

    while (exitLoop == false) {
      while ( radio.available(&pipeNo)) {
        radio.read( &risposta, sizeof(unsigned int) );
        //Serial.println(">>>> ricevuto qualcosa");
        if (risposta == 91) {
          Serial.println(">>> Fotocellula start presente!");
          exitLoop = true;
          Stato = 2;
          break;
        }
        else {
          count ++;
          if (count >= 3) {
            exitLoop = true;
            Stato = 5;
            break;
          }
        }
      }
    }
  }
}

void startTime(void) {

  radio.stopListening();
  Serial.println("*** Attivare fotocellula start");

  unsigned int risposta;

  if (!radio.write( &Stato, sizeof(unsigned int) )) {
    Serial.println(">>> Errore nel invio dello stato start");
    Stato = 5;
    radio.startListening();
  } else {

    if (!radio.available()) {
      radio.startListening();

      byte pipeNo;
      bool exitLoop = false;
      int count = 0;

      while (exitLoop == false) {
        while ( radio.available(&pipeNo)) {
          radio.read( &risposta, sizeof(unsigned int) );
          //Serial.println(">>>> ricevuto qualcosa");
          if (risposta == 92) {
            t0 = millis();
            Serial.println(">>> Segnale start ricevuto!");
            exitLoop = true;
            Stato = 3;
            break;
          }
          else {
            count ++;
            if (count >= 3) {
              exitLoop = true;
              Stato = 5;
              break;
            }
          }
        }
      }
    } else {
      Serial.println(">>> ERRORE");
    }
  }
}

void stopTime(void) {

  Serial.println("*** Attivare fotocellula stop");

  while ((digitalRead(sensorPin) == HIGH) && ((millis() - t0) < 5000)) {
    //Aspetta l'interruzione della fotocellula stop
  }
  t1 = millis();

  if ((t1 - t0) >= 5000) {
    Stato = 5;
    Serial.println(">>> ERRORE STOP");
  }
  else {
    Stato = 4;
    Serial.println(">>> STOP ricevuto");
  }
}

void displayTime(void) {

  inviaValoreADisplay(t1 - t0);
}

void errore(void) {
  inviaValoreADisplay(8888);
}

void inviaValoreADisplay(long int valore) {

  radio.stopListening();

  radio.openWritingPipe(pipes[2]);

  if (!radio.write( &valore, sizeof(long int) )) {
    Serial.println(">>> Errore nel invio al display del valore richiesto");
  } else {
    Serial.print("*** Valore (");
    Serial.print(valore);
    Serial.println(") inviato correttamente al display");
  }

  //Alla fine ristabilire i Pipe normali di comunicazione tra le fotocellule
  if (role == role_start) {
    radio.openWritingPipe(pipes[1]);
  }
  else {
    radio.openWritingPipe(pipes[0]);
  }

  radio.startListening();
}


void attesaPronto(void) {

  Serial.println("*** Attesa del pronto dalla fotocellula stop");

  unsigned int risposta;
  byte pipeNo;
  bool exitLoop = false;

  while (exitLoop == false) {
    //Serial.println(">>>> loop");
    while ( radio.available(&pipeNo)) {
      radio.read( &risposta, sizeof(unsigned int) );
      //Serial.println(">>>> ricevuto qualcosa");
      if (risposta == 1) {
        risposta = 91;                            //Fotocellula start pronta a ricevere il comando per inviare lo start
        Serial.println("*** Invio risposta pronto alla fotocellula stop");
        radio.stopListening();
        if (!radio.write( &risposta, sizeof(unsigned int) )) {
          Serial.println(">>> Errore nel invio dello stato pronto");
          radio.startListening();
        }
        else {

          if (!radio.available()) {
            Serial.println(">>> Ricevuto ACK dalla fotocellula stop");
          } else {
            Serial.println(">>> ERRORE nessun ACK ricevuto");
          }

          radio.startListening();
        }

        exitLoop = true;
        break;
      }
      //radio.writeAckPayload(pipeNo,&risposta, sizeof(unsigned int) );


    }
  }
}

void attesaStart(void) {


  Serial.println("*** Attesa del segnale di start da trasmettere");

  unsigned int risposta;
  byte pipeNo;
  bool exitLoop = false;

  while (exitLoop == false) {
    //Attesa del ok per inviare il segnale di start
    while ( radio.available(&pipeNo)) {
      radio.read( &risposta, sizeof(unsigned int) );
      if (risposta == 2) {
        Serial.println(">>> Invio sistema pronto a ricevere il segnale di start dalla fotocellula");
        exitLoop = true;
        break;
      }
    }
  }

  Serial.println("*** Lettura fotocellula start");
  //Attesa del interruzione della fotocellula
  risposta = 92;
  while (digitalRead(sensorPin) == HIGH) {
    //Attessa, nessuna azione
  }
  Serial.println("*** Invio segnale di start");
  //Inviare segnale di start alla fotocellula stop
  radio.stopListening();
  if (!radio.write( &risposta, sizeof(unsigned int) )) {
    Serial.println(">>> Errore nel invio del segnale di start");
  }

  radio.startListening();

}


